<?php

namespace App\Http\Controllers\Auth;

use App\Helpers\ApiResponse;
use App\Http\Controllers\Controller;
use App\Http\Requests\Auth\LoginRequest;
use App\Http\Requests\Auth\RegisterRequest;
use App\Http\Resources\UserResource;
use Illuminate\Http\Request;
use App\Models\User;
use Exception;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Storage;
use Illuminate\Validation\ValidationException;

class AuthController extends Controller
{
    public function register(RegisterRequest $request)
    {
        try {
            $validated = $request->validated();

            $imagePath = 'https://i.pravatar.cc/300';
            if ($request->hasFile('avatar')) {
                $imagePath = $request->file('avatar')->store('users', 'public');
            }

            // Create the user
            $user = User::create([
                'name'       => $validated['name'],
                'email'      => $validated['email'],
                'password'   => Hash::make($validated['password']),
                'role'       => 'user',
                'location'   => $validated['location'],
                'age'        => $validated['age'],
                'image_url'  => $imagePath,
            ]);

            // Sanctum Token
            $token = $user->createToken('auth_token')->plainTextToken;

            return ApiResponse::success(
                "Registration successful!",
                [
                    'user'  => $user,
                    'token' => $token
                ],
                201
            );

        } catch (Exception $e) {
            // Unexpected error
            return ApiResponse::error(
                "Something went wrong: " . $e->getMessage(),
                500
            );
        }
    }


    public function login(LoginRequest $request)
    {
        try {
            $validated = $request->validated();

            $user = User::where('email', $validated['email'])->first();

            if (!$user) {
                return ApiResponse::error("User not found", 404);
            }

            if (!Hash::check($validated['password'], $user->password)) {
                return ApiResponse::error("Incorrect password", 401);
            }

            // Create Sanctum Token
            $token = $user->createToken('auth_token')->plainTextToken;

            return ApiResponse::success(
                "Login successful",
                [
                    'user'  => new UserResource($user),
                    'token' => $token
                ]
            );

        } catch (Exception $e) {
            return ApiResponse::error("Something went wrong: " . $e->getMessage(), 500);
        }
    }


    public function logout(Request $request)
    {
        try {
            // Delete only the current access token
            $request->user()->currentAccessToken()->delete();

            return ApiResponse::success("Logout successful");
        } catch (Exception $e) {
            return ApiResponse::error("Logout failed: " . $e->getMessage(), 500);
        }
    }
}







