<?php

namespace App\Http\Controllers;

use App\Helpers\ApiResponse;
use App\Http\Requests\TravelPlan\StoreTravelPlanRequest;
use App\Http\Requests\TravelPlan\UpdateTravelPlanRequest;
use App\Http\Resources\InterestResource;
use App\Http\Resources\TravelPlanResource;
use App\Models\TravelPlan;
use App\Models\TravelPlanParticipant;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;

class TravelPlanController extends Controller
{
    // GET /api/travel-plans?page=1&user_id=123 (user_id optional)
    public function index(Request $request): JsonResponse
    {
        $query = TravelPlan::query()->orderBy('start_date', 'desc');

        if ($request->filled('user_id')) {
            $query->where('user_id', $request->integer('user_id'));
        }

        $plans = $query->paginate(10);

        return ApiResponse::success("Travel plans fetched successfully", $plans);
    }

    // POST /api/travel-plans
    public function store(StoreTravelPlanRequest $request): JsonResponse
    {
        $data = $request->validated();

        $plan = TravelPlan::create([
            'user_id'     => $data['user_id'],
            'title' => $data['title'],
            'short_description' => $data['short_description'] ?? null,
            'destination' => $data['destination'],
            'start_date'  => $data['start_date'],
            'end_date'    => $data['end_date'],
            'budget'      => $data['budget'] ?? null,
            'travel_type' => $data['travel_type'] ?? null,
            'itinerary' => $data['itinerary'] ?? [],
            'group_size'  => $data['group_size'] ?? 1,
            'status'      => $data['status'],
        ]);

        return ApiResponse::success("Travel plan created successfully", $plan, 201);
    }

    // GET /api/travel-plans/{id}
    public function show(TravelPlan $travelPlan): JsonResponse
    {
        // $travelPlan->load('host','participants', 'reviews.reviewer');
        $travelPlan->load('host','participants');

        // $participants = $travelPlan->participants;

        // $exists = $participants->contains('user_id', $userId);

        return ApiResponse::success(
            "Travel plan details",
            new TravelPlanResource($travelPlan)
        );
    }

    public function upcomingTrips()
    {
        $upcomingTrips = TravelPlan::where('start_date', '>', now())
            ->where('status', 'active')
            ->orderBy('start_date', 'asc')
            ->get();

        return ApiResponse::success(
            "Upcoming trips fetched successfully",
            $upcomingTrips
        );
    }

    // PUT/PATCH /api/travel-plans/{id}
    public function update(UpdateTravelPlanRequest $request, TravelPlan $travelPlan): JsonResponse
    {
        $data = $request->validated();
        $travelPlan->update($data);

        return ApiResponse::success("Travel plan updated successfully", $travelPlan);
    }

    // DELETE /api/travel-plans/{id}
    public function destroy(TravelPlan $travelPlan): JsonResponse
    {
        $travelPlan->delete();

        return ApiResponse::success("Travel plan deleted successfully");
    }




    /**
     * ================ Travel Plan Participant Controller ==================
     * User requests to join a travel plan
     */
    public function join(Request $request, $id)
    {
        $user = auth()->user();

        // return response()->json($request->all());

        // Check if travel plan exists
        $travelPlan = TravelPlan::find($id);

        if (!$travelPlan) {
            return ApiResponse::error(
                'Travel plan not found',
                404
            );
        }

        // Host can't join own plan
        if ($travelPlan->user_id == $user->id) {
            return ApiResponse::error(
                'You are the host of this trip',
                403
            );
        }

        // Prevent duplicate request
        $existing = TravelPlanParticipant::where('travel_plan_id', $id)
                    ->where('user_id', $user->id)
                    ->first();

        if ($existing) {
            return ApiResponse::error('You already sent a request',400);
        }

        // Create pending request
        $travelPlanParticipant = TravelPlanParticipant::create([
            "travel_plan_id" => $id,
            "user_id" => $user->id,
            "role" => "member",
            "status" => "pending"
        ]);

        return ApiResponse::success(
            "Your request has been sent to the host",
            $travelPlanParticipant
        );
    }


    /**
     * Host approves join request
     */
    public function approve(Request $request, $id)
    {
        $participant = TravelPlanParticipant::find($id);

        if (!$participant) {
            return response()->json(["success" => false, "message" => "Request not found"], 404);
        }

        // Only host can approve
        if ($participant->travelPlan->user_id !== auth()->id()) {
            return response()->json(["success" => false, "message" => "Unauthorized"], 403);
        }

        $participant->update(["status" => "approved"]);

        return response()->json(["success" => true, "message" => "User approved"]);
    }

    /**
     * Host declines join request
     */
    public function decline(Request $request, $id)
    {
        $participant = TravelPlanParticipant::find($id);

        if (!$participant) {
            return response()->json(["success" => false, "message" => "Request not found"], 404);
        }

        if ($participant->travelPlan->user_id !== auth()->id()) {
            return response()->json(["success" => false, "message" => "Unauthorized"], 403);
        }

        $participant->update(["status" => "declined"]);

        return response()->json(["success" => true, "message" => "User declined"]);
    }


    // =============== Search & Match ================
    public function match(Request $request)
    {
        // return response()->json(['status' =>'ok']);

        $destination = $request->destination;      // string
        $startDate   = $request->start_date;       // YYYY-MM-DD
        $endDate     = $request->end_date;         // YYYY-MM-DD
        $interestIds = $request->filled('interests')
            ? explode(',', $request->interests)    // "9,23,24" → [9,23,24]
            : [];

        // Base query with relations
        $query = TravelPlan::query()
            ->with(['host', 'participants', 'interests']);

        // If at least one filter present, wrap in OR group
        if ($destination || ($startDate && $endDate) || !empty($interestIds)) {

            $query->where(function ($q) use ($destination, $startDate, $endDate, $request) {

                // OR: Destination match
                if ($destination) {
                    $q->orWhere('destination', 'LIKE', "%{$destination}%");
                }

                // OR: Date range overlap
                if ($startDate && $endDate) {
                    // Example logic: trip's date range overlaps given range
                    $q->orWhere(function ($sub) use ($startDate, $endDate) {
                        $sub->whereDate('start_date', '<=', $endDate)
                            ->whereDate('end_date', '>=', $startDate);
                    });
                }

                // OR: Interests match (plan interests)
                if ($request->filled('interests')) {
                    $interestNames = explode(',', $request->interests);

                    $q->orWhereHas('interests', function ($q) use ($interestNames) {
                        $q->whereIn('name', $interestNames);
                    });
                }


            });
        }

        // Paginate & transform response
        $plans = $query
            ->orderBy('start_date', 'asc')
            ->paginate(10)
            ->through(function ($plan) {
                return [
                    'id'          => $plan->id,
                    'title'       => $plan->title,
                    'destination' => $plan->destination,
                    'start_date'  => $plan->start_date,
                    'end_date'    => $plan->end_date,
                    'budget'      => $plan->budget,
                    'group_size'  => $plan->group_size,
                    'status'      => $plan->status, // active / completed / draft etc.

                    // Host (trip owner)
                    'host' => $plan->host ? [
                        'id'       => $plan->host->id,
                        'name'     => $plan->host->name,
                        'location' => $plan->host->location,
                        'imageUrl' => isset($plan->host->image_url)
                            && Storage::disk('public')->exists($plan->host->image_url)
                                ? url("storage/{$plan->host->image_url}")
                                : $plan->host->image_url,
                        'rating_avg'   => $plan->host->rating_avg ?? 0,
                        'rating_count' => $plan->host->rating_count ?? 0,
                    ] : null,

                    // Plan interests
                    'interests' => InterestResource::collection($plan->interests),

                    // Participants info
                    'participants_count' => $plan->participants ? $plan->participants->count() : 0,
                ];
            });

        return ApiResponse::success(
            "Matched travel plans fetched successfully!",
            $plans
        );
    }
}
