<?php

namespace App\Http\Controllers;

use App\Http\Requests\User\StoreUserRequest;
use Illuminate\Http\Request;
use App\Models\User;
use Illuminate\Http\JsonResponse;
use App\Helpers\ApiResponse;
use App\Http\Requests\User\UpdateUserRequest;
use App\Http\Resources\InterestResource;
use App\Http\Resources\UserResource;
use Illuminate\Support\Facades\Storage;
class UserController extends Controller
{
    public function index()
    {
        $users = User::orderBy('id', 'desc')->paginate(10)
            ->through(function ($user) {
                return [
                    'id'=> $user->id,
                    'name'=> $user->name ?? null,
                    'email'=> $user->email ?? null,
                    'bio'=> $user->bio ?? null,
                    'imageUrl'=> isset($user->image_url) && Storage::disk('public')->exists($user->image_url) ? url("public/storage/{$user->image_url}") : null,
                    'location'=> $user->location ?? null,
                ];
            });


        return ApiResponse::success(
            "User Fetch successfully!",
            $users
        );
    }

    public function store(StoreUserRequest $request): JsonResponse
    {
        $data = $request->validated();

        // image optional
        if ($request->hasFile('image')) {
            $imagePath = $request->file('image')->store('users', 'public');
        }

        // adapt fields to your users table
        $user = User::create([
            'name'     => $data['name'],
            'email'    => $data['email'],
            'password' => bcrypt('password'), // temp, or change later
            'location' => $data['location'] ?? null,
            'bio'      => $data['bio'] ?? null,
            // You can store interests/countries in JSON fields if you have them:
            // 'interests' => $data['interests'] ?? [],
            // 'countries' => $data['countries'] ?? [],
            // 'image_url' => $data['image'] ?? null,
            // 'interests' => json_encode($validated['interests'] ?? []),
            // 'countries' => json_encode($validated['countries'] ?? []),
            'image_url' => $imagePath ?? null,
        ]);

        return ApiResponse::success(
            "User Created successfully!",
            $user
        );
    }

    public function show($id)
    {
        try {
            $user = User::find($id);

            if (!$user) {
                return ApiResponse::error('User not found', 404);
            }

            return ApiResponse::success(
                "User fetched successfully!",
                new UserResource($user)
            );

        } catch (\Exception $e) {
            return ApiResponse::error(
                "Something went wrong",
                500
            );
        }
    }


    public function update(UpdateUserRequest $request, $id)
    {

        // return response()->json($request->interests);


        $user = User::find($id);

        if (!$user) {
            return ApiResponse::error(
                'User not found',
                404
            );
        }

        $data = $request->validated();

        // Image handling (optional)
        if ($request->hasFile('image')) {
            $path = $request->file('image')->store('users', 'public');
            $data['image_url'] = $path;
        }

        try {
            $user->update($data);

            $interests = $request->interests;
            $interestIds = is_string($interests) ? json_decode($interests, true): $interests;
            $interestIds = is_array($interestIds) ? $interestIds : [];
            $user->interests()->sync($interestIds);

        } catch (\Exception $e) {
            return ApiResponse::error(
                "Something went wrong: " . $e->getMessage(),
                500
            );
        }



        return ApiResponse::success(
            "User updated successfully",
            new UserResource($user)
        );
    }


    public function destroy($id)
    {
        $user = User::find($id);

        if (!$user) {
            return ApiResponse::error(
                'User not found',
                404
            );
        }

        $user->delete();

        return ApiResponse::success(
            "User deleted successfully",
            User::orderBy('id', 'desc')->get()
        );
    }

    // public function getUserWithInterest()
    // {
    //     $users = User::with('interests')->orderBy('id', 'desc')->paginate(10)
    //         ->through(function ($user) {
    //             return [
    //                 'id'=> $user->id,
    //                 'name'=> $user->name ?? null,
    //                 'email'=> $user->email ?? null,
    //                 'role'  => $user->role,
    //                 'bio'  => $user->bio,
    //                 'imageUrl' => isset($user->image_url) && Storage::disk('public')->exists($user->image_url) ? url("storage/{$user->image_url}") : $user->image_url,
    //                 'location'  => $user->location,
    //                 'age'  => $user->age,
    //                 'rating_avg'  => $user->rating_avg,
    //                 'rating_count'  => $user->rating_count,
    //                 'interests' => InterestResource::collection($user->interests),
    //             ];
    //         });


    //     return ApiResponse::success(
    //         "User with interest fetch successfully!",
    //         $users
    //     );
    // }

    public function getUserWithInterest(Request $request)
    {
        $query = User::query()->with('interests');

        // 🔥 Optional filtering by interest ID (OR matching)
        if ($request->filled('interests')) {
            $interestIds = explode(',', $request->interests);

            $query->whereHas('interests', function ($q) use ($interestIds) {
                $q->whereIn('interest_id', $interestIds);  // OR match
            });
        }

        // 🔥 Filter by destination
        if ($request->filled('location')) {
            $location = $request->location;

            $query->where('location', 'LIKE', "%{$location}%");
        }

        // 🔥 Age filter (min_age + max_age both optional)
        if ($request->filled('min_age')) {
            $query->where('age', '>=', (int)$request->min_age);
        }

        if ($request->filled('max_age')) {
            $query->where('age', '<=', (int)$request->max_age);
        }


        $users = $query->orderBy('id', 'desc')
            ->paginate(10)
            ->through(function ($user) {
                return [
                    'id' => $user->id,
                    'name' => $user->name ?? null,
                    'email' => $user->email ?? null,
                    'role' => $user->role,
                    'bio' => $user->bio,
                    'imageUrl' => isset($user->image_url) &&
                                Storage::disk('public')->exists($user->image_url)
                                    ? url("public/storage/{$user->image_url}")
                                    : $user->image_url,
                    'location' => $user->location,
                    'age' => $user->age,
                    'rating_avg' => $user->rating_avg,
                    'rating_count' => $user->rating_count,
                    'interests' => InterestResource::collection($user->interests),
                ];
            });

        return ApiResponse::success(
            "User with interest fetch successfully!",
            $users
        );
    }

}
